' Generic Solitaire that supports many different games
' Rev 1.0.0 William M Leue 9-Mar-2022

option default integer
option base 1

const NSUITS = 4
const NRANKS = 13
const NCARDS = NSUITS*NRANKS
const ACE = 1
const TWO = 2
const THREE = 3
const FOUR = 4
const FIVE = 5
const SIX = 6
const SEVEN = 7
const EIGHT = 8
const NINE = 9
const TEN = 10
const JACK = 11
const QUEEN = 12
const KING = 13

const SPADES = 1
const HEARTS = 2
const DIAMONDS = 3
const CLUBS = 4
const FACEUP = 1
const FACEDOWN = 0

' Colors
const SELECTED_COLOR = RGB(CYAN)
const UNSELECTED_COLOR = RGB(BLACK)
const SOURCE_COLOR = RGB(MAGENTA)
const DESTINATION_COLOR = RGB(GREEN)

' More graphic constants
const CARD_WIDTH = 74
const CARD_HEIGHT = 94
const CARD_CRAD   = 10
const CARD_HJOG = 10
const CARD_VJOG = 20
const TABLE_COLOR = RGB(0, 130, 0)
const CEDGE_COLOR = RGB(BLACK)
const LEFT_MARGIN = 20
const TOP_MARGIN = 50
const CARD_SPACING = 10
const CARD_VSPACING = 17
const CBACK_SOURCE_INDEX = 53

' mouse channel
const MCHAN = 2

' keyboard keys
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const F1    = 145
const ESC   = 27

' Limits on piles
const MAX_DECKS = 2
const MAX_FOUNDATIONS = 8
const MAX_TABLEAUX = 16
const MAX_RESERVE = 8
const MAX_PILES = 2+MAX_FOUNDATIONS+MAX_TABLEAUX+MAX_RESERVE

' solitaire definition params
const NLAYOUT_PARS  = 4
const NREMOVE_RULES = 2
const NADD_RULES    = 7
const NDEAL_PARS    = 2
const NPILEPARS = 1+NLAYOUT_PARS+NREMOVE_RULES+NADD_RULES+NDEAL_PARS
const RM_ALL_CARDS = 99

' Game selection window params
const GS_WIDTH = 200
const GS_HEIGHT = 60
const GS_X = mm.hres\2 - GS_WIDTH
const GS_Y = 100

' pile info components
' There are 5 sections:
'  pile role (1 value)
'  pile location and layout (4 values)
'  card remove rules (2 values)
'  card add rules (7 values)
'  pile intial deal (2 values)
const P_ROLE = 1  ' 1=Deck,2=Discard,3=Foundation,4=Tableau,5=Reserve
const P_COL  = 2  ' Column location (1..n)
const P_ROW  = 3  ' Row location (1..n)
const P_ST_X = 4  ' Horizontal stacking rule
const P_ST_Y = 5  ' vertical stacking rule
const P_RM_N = 6  ' max # cards to remove at once
const P_RM_F = 7  ' face up/down 
const P_AD_M = 8  ' max # cards in pile
const P_AD_I = 9  ' initial  card rank
const P_AD_N = 10 ' max # cards to add at once
const P_AD_F = 11 ' face up/down
const P_AD_R = 12 ' rank following rule
const P_AD_W = 13 ' ranks wrap?
const P_AD_S = 14 ' suit/color following rule
const P_DL_D = 15 ' initial deal # face down
const P_DL_U = 16 ' initial deal # face up
 
' pile role values
const ROLE_DECK       = 1
const ROLE_DISCARD    = 2
const ROLE_FOUNDATION = 3
const ROLE_TABLEAU    = 4
const ROLE_RESERVE    = 5
const MAX_DISCARDS    = 3

' Control Buttons
const BUTTON_UNDO     = 1
const BUTTON_NEWGAME  = 2
const BUTTON_GAMES    = 3
const BUTTON_RULES    = 4
const BUTTON_QUIT     = 5
const NBUTTONS        = 5

const DEBUG = 1

' remove cards rejection reason
const RR_NUM          = 1
const RR_FACE         = 2

' add cards rejection reasons
const AR_NOCARDS      = 1
const AR_MAXTOTAL     = 2
const AR_BATCHSIZE    = 3
const AR_INITRANK     = 4
const AR_FACE         = 5
const AR_DECRANK      = 6
const AR_INCRANK      = 7
const AR_SAMERANK     = 8
const AR_SUIT         = 9
const AR_ALTCRB       = 10
const AR_ALTCBR       = 11

' Solitaire Types
const KLONDIKE          = 1
const CANFIELD          = 2
const EIGHTOFF          = 3
const PILE_ON           = 4
const YUKON             = 5
const GOLF              = 6
const BRISTOL           = 7
const FREECELL          = 8
const FOUR_SEASONS      = 9
const BELEAGURED_CASTLE = 10
const NSOLS             = 10
const MAX_SOLS          = 10

const MAX_RULES_LINES = 10

' Globals
dim num_decks = 1
dim num_discards = 1
dim num_cards = NCARDS
dim num_foundations = 1
dim num_tableaux = 1
dim num_reserve = 1
dim num_piles = 0
dim pile_info(NPILEPARS, MAX_PILES)
dim running = 0
dim source(56,2)
dim deck(num_cards)
dim deck_pindex = 0
dim discard_pindex(MAX_DISCARDS)

' game type params
dim snames$(MAX_SOLS)
dim solitaire_type = 7
dim rules$(MAX_RULES_LINES)
dim num_rule_lines = 0
dim gs_row = 0
dim gs_col = 0
dim gs_active = 0

' these get redimensioned after info file is read
dim piles(2)
dim pnum_cards(2)
dim bboxes(4, 2)

' deck for restart
dim restart_deck(num_cards)

' game state machine variables
dim left_click = 0
dim right_click = 0
dim prev_left_mx = 0
dim prev_left_my = 0
dim left_mx = 0
dim left_my = 0
dim left_busy = 0
dim prev_right_mx = 0
dim prev_right_my = 0
dim right_mx = 0
dim right_my = 0
dim right_busy = 0
dim selected_pile = 0
dim source_pile = -1
dim destination_pile = -1
dim move_state = 0

' button locations
dim button_x(NBUTTONS) = (40, 200, 360, 520, 680)
dim button_y = 5
dim button_w = 100
dim button_h = 30

' debug stuff
dim rm_reason = 0
dim add_reason = 0

' Main program
open "debug.txt" for output as #1
PrepGraphics
InitMouse
GetCardImages
'ShowStartScreen
ReadSolitaireNames
ShowGameMenu
running = 1
PlayLoop
end

' set up mode, clear pages
sub PrepGraphics
  mode 1, 8
  page write 1
  cls
  page write 0
  cls
end sub

' Read the names of the supported game types
sub ReadSolitaireNames
  local i
  for i = 1 to MAX_SOLS
    read snames$(i)
  next i
end sub

' Initialize mouse and  cursor
sub InitMouse
  controller mouse open MCHAN, LClick, RClick
  gui cursor on 1, 10, 10, rgb(red)
  gui cursor show
  settick 20, UpdateCursor
end sub

' Have the cursor track the mouse
sub UpdateCursor
  gui cursor mouse(X), mouse(Y)
end sub

' Update game state machine on left mouse button click
sub LClick
  if left_click = 1 then exit sub
  prev_left_mx = left_mx : prev_left_my = left_my
  left_mx = mouse(X) : left_my = mouse(Y)
  left_click = 1
  left_busy = 0
end sub

' Update game state machine on right mouse button click
sub RClick
  if right_click = 1 then exit sub
  prev_right_mx = right_mx : prev_right_my = right_my
  right_mx = mouse(X) : right_my = mouse(Y)
  right_click = 1
  right_busy = 0
end sub

' Event handler for play. Detects mouse clicks and keyboard
' presses, and dispatches according to the current game state
' The left_click, right_click, left_busy, and right_busy globals
' coordinate mouse action and game state.
sub PlayLoop
  local z$
  local cmd, pile, nsc, nc, cx, i
  z$ = INKEY$
  do
    ' ---- left buttton click
    if left_click = 1 and left_busy = 0 then
      left_busy = 1
      left_click = 0
      if HandleButtons(left_mx, left_my) then
        left_busy = 0
        continue do
      end if
      if not running then
        left_busy = 0
        continue do
      end if
      for pile = 1 to num_piles
        if left_mx >= bboxes(1, pile) and left_mx <= bboxes(3, pile) then
          if left_my >= bboxes(2, pile) and left_my <= bboxes(4, pile) then
            if pile = deck_pindex then
              DealCardsFromDeck
            else
              SelectPile pile, left_my
            end if
            left_busy = 0
            continue do
          end if
        end if
      next pile
      ClearMovePiles
      left_busy = 0
    end if
    ' ----- right button click
    if right_click = 1 and right_busy = 0 then
      right_busy = 1
      right_click = 0
      if not running then
        right_busy = 0
        continue do
      end if
      if source_pile > 0 then
        nsc = pnum_cards(source_pile)
        if nsc > 0 then
          nc = GetNumCardsToMove(source_pile, right_mx, right_my)
          if CanRemove(source_pile, 0, nsc, nc) then
            cx = nsc-nc+1
            for i = 1 to num_piles
              if i <> source_pile then
                if CanAdd(i, source_pile, cx, nc) then
                  MoveCards source_pile, i, nc
                  DrawPile source_pile
                  DrawPile i
                  exit for
                end if
              end if
            next i
          end if
        end if
      end if
      ClearMovePiles
      right_busy = 0
    end if
    ' ----- keyboard
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case UP
          if gs_active then
            inc gs_row, -1
            if gs_row < 1 then gs_row = MAX_SOLS\2
            HiliteGame gs_row, gs_col
          end if
        case DOWN
          if gs_active then
            inc gs_row
            if gs_row > MAX_SOLS\2 then gs_row = 1
            HiliteGame gs_row, gs_col
          end if
        case LEFT
          if gs_active then
            gs_col = 3 - gs_col
            HiliteGame gs_row, gs_col
          end if
        case RIGHT
          if gs_active then
            gs_col = 3 - gs_col
            HiliteGame gs_row, gs_col
          end if
        case ENTER
          if gs_active then
            solitaire_type = (gs_col-1)*MAX_SOLS\2 + gs_row
            gs_active = 0
            gui cursor show
            ReadSolitaireDefFile "./Games/" + snames$(solitaire_type) + ".sol"
            NewGame
            DrawTable
          end if
        case F1
          ShowInstructions
          if gs_active then ShowGameMenu else DrawTable
        case ESC
          Quit
      end select
    end if
  loop
end sub

sub Quit
  settick 0, 0
  gui cursor hide
  gui cursor off
  controller mouse close
  cls
  end
end sub

' Read a Solitaire Definition File
sub ReadSolitaireDefFile path$
  local buf$, p1$, p2$, p3$, p4$, p5$
  local i, n, p, st
  on error skip 1
  open path$ for input as #2
  if mm.errno <> 0 then
    print "Error opening solitaire def file"
    end
  end if
  line input #2, buf$
  st = val(buf$)
  line input #2, buf$
  num_decks = val(buf$)
  line input #2, buf$
  num_discards = val(buf$)
  line input #2, buf$
  num_foundations = val(buf$)
  line input #2, buf$
  num_tableaux = val(buf$)
  line input #2, buf$
  num_reserves = val(buf$)
  num_piles = num_decks+num_discards+num_foundations+num_tableaux+num_reserves
  for p = 1 to num_piles
    line input #2, buf$
    p1$ = field$(buf$, 1, ";")
    p2$ = field$(buf$, 2, ";") + ","
    p3$ = field$(buf$, 3, ";") + ","
    p4$ = field$(buf$, 4, ";") + ","
    p5$ = field$(buf$, 5, ";") + ","
    k = val(p1$)
    pile_info(P_ROLE, p) = val(p1$)
    pile_info(P_COL, p)  = val(field$(p2$, 1, ","))
    pile_info(P_ROW, p)  = val(field$(p2$, 2, ","))
    pile_info(P_ST_X, p) = val(field$(p2$, 3, ","))
    pile_info(P_ST_Y, p) = val(field$(p2$, 4, ","))
    pile_info(P_RM_N, p) = val(field$(p3$, 1, ","))
    pile_info(P_RM_F, p) = val(field$(p3$, 2, ","))
    pile_info(P_AD_M, p) = val(field$(p4$, 1, ","))
    pile_info(P_AD_I, p) = val(field$(p4$, 2, ","))
    pile_info(P_AD_N, p) = val(field$(p4$, 3, ","))
    pile_info(P_AD_F, p) = val(field$(p4$, 4, ","))
    pile_info(P_AD_R, p) = val(field$(p4$, 5, ","))
    pile_info(P_AD_W, p) = val(field$(p4$, 6, ","))
    pile_info(P_AD_S, p) = val(field$(p4$, 7, ","))
    pile_info(P_DL_D, p) = val(field$(p5$, 1, ","))
    pile_info(P_DL_U, p) = val(field$(p5$, 2, ","))
  next p
  num_rule_lines = 0
  line input #2, buf$
  do while buf$ <> ""
    inc num_rule_lines
    rules$(num_rule_lines) = buf$
    line input #2, buf$
  loop
  close #2
end sub

' Make a new game
sub NewGame
  local i
  CreatePiles
  MakeDeck
  ShuffleDeck
  select case solitaire_type
    case BELEAGURED_CASTLE
      DoSpecialStartRules
  end select
  DealInitialCards
  select case solitaire_type
    case CANFIELD
      DoSpecialStartRules
    case FOUR_SEASONS
      DoSpecialStartRules
  end select
  sp = 0
  running = 1
end sub

' Create the piles of cards from the solitaire definition
' Pile-related storage is re-dimensioned to match # of piles
' for the current solitaire game.
sub CreatePiles
  local p, role, dp, dn
  erase piles()
  erase pnum_cards()
  erase bboxes()
  dim piles(num_cards, num_piles)
  dim pnum_cards(num_piles)
  dim bboxes(4, num_piles)
  dn = 0
  for p = 1 to num_piles
    pnum_cards(p) = 0
    role = pile_info(P_ROLE, p)
    if role = ROLE_DECK then deck_pindex = p
    if role = ROLE_DISCARD then
      inc dn
      discard_pindex(dn) = p
    end if
    AdjustBoundingBox p
  next p
end sub

' Load the deck with a pack of cards in
' standard order, all face-down. (Note, CreatePiles
' method MUST be called before calling this.)
sub MakeDeck
  local suit, rank, n
  if deck_pindex = 0 then
    print "Error - invalid deck_pindex"
    end
  end if
  n = 0
  for suit = SPADES to CLUBS
    for rank = ACE to KING
      inc n
      piles(n, deck_pindex) = EncodeCard(rank, suit, FACEDOWN)
      next rank
    next suit
  pnum_cards(deck_pindex) = num_cards
end sub

' Shuffle the deck
sub ShuffleDeck
  local n, b, temp
  for n = 1 to NCARDS
    b = max(1,min(rnd*NCARDS, NCARDS))
    temp = piles(n, deck_pindex)
    piles(n, deck_pindex) = piles(b, deck_pindex)
    piles(b, deck_pindex) = temp
  next n
  for n = 1 to NCARDS
    restart_deck(n) = piles(n, deck_pindex)
  next n
end sub

' Deal the initial cards as per the solitaire definition
sub DealInitialCards
  local p, nu, nd, nc, i
  for p = 1 to num_piles
    nd = pile_info(P_DL_D, p)
    nu = pile_info(P_DL_U, p)
    nc = nd+nu
    for i = 1 to nc
      if i <= nd then
        PredealCard FACEDOWN, p
      else
        PredealCard FACEUP, p
      end if
    next i
  next p
end sub

' Deal a card from the Deck to a pile
' Card gets flipped if requested FACEUP.
' This is only used for predeals at start of game!
sub PredealCard face, p
  local card, rank, suit, cface
  RemoveCardFromPile deck_pindex, -1, card
  DecodeCard card, rank, suit, cface
  card = EncodeCard(rank, suit, face)
  end if
  AddCardToPile card, p
end sub

' Do any special start game rules for specific solitaire types
sub DoSpecialStartRules
  select case solitaire_type
    case KLONDIKE
    case CANFIELD
      DoRankStartSpecial
    case EIGHTOFF
    case SCORPION
    case YUKON
    case BRISTOL
    case FREECELL
    case FOUR_SEASONS
      DoRankStartSpecial
    case BELEAGURED_CASTLE
      DoBeleaguredCastleStartSpecial
  end select
end sub

' Do special start rules for Canfield and FourSeasons. These must be done
' AFTER the initial predeal.
sub DoRankStartSpecial
  local i, sf, ef, card, rank, suit, face
  sf = 3 : ef = 6
  card = piles(1, sf)
  DecodeCard card, rank, suit, face
  for i = sf to ef
    pile_info(P_AD_I, i) = rank
  next i
end sub

' Do special start rules for Beleagured Castle
' These must be done BEFORE the initial predeal.
sub DoBeleaguredCastleStartSpecial
  local i, card, fx, rank, suit, face, ndc, n
  local ax(4)
  fx = -1
  for i = 1 to num_piles
    if pile_info(P_ROLE, i) = ROLE_FOUNDATION then
      fx = i
      exit for
    end if
  next i
  ndc = pnum_cards(deck_pindex)
  n = 0
  for i = 1 to ndc
    card = piles(i, deck_pindex)
    DecodeCard card, rank, suit, face
    if rank = ACE then
      inc n
      ax(n) = i
    end if
  next i
  for i = n to 1 step -1
    RemoveCardFromPile deck_pindex, ax(i), card
    AddCardToPile card, fx
    TurnTopCard fx, FACEUP
    inc fx
  next i
end sub

' Remove a card from a pile. If not the last card, move
' the cards above it to fill its space. Adjust the pile
' bounding box. (Note: this does not check legality. The CanRemove()
' function must be called first to determine legality.
sub RemoveCardFromPile pindex, cindex, card
  local nc, findex, tindex, rank, suit, face
  if pindex = 0 or cindex = 0 then exit sub
  nc = pnum_cards(pindex)
  if cindex < 1 then cindex = nc
  card = piles(cindex, pindex)
  if cindex < nc then
    for i = cindex+1 to nc
      piles(i-1, pindex) = piles(i, pindex)
    next i
  end if
  inc pnum_cards(pindex), -1
  ClearBoundingBox pindex
  AdjustBoundingBox pindex
  if pile_info(P_ROLE, pindex) = ROLE_TABLEAU then
    if nc > 1 then
      TurnTopCard pindex, FACEUP
    end if
  end if
end sub

' Add a card to a pile and adjust its bounding box
' Added cards always go on top of any existing cards.
' (Note: this does not check legality. The CanAdd() function
' must be called first to determine legality.
sub AddCardToPile card, pindex
  local nc = pnum_cards(pindex)
  piles(nc+1, pindex) = card
  inc pnum_cards(pindex), 1
  AdjustBoundingBox pindex
end sub

' When cards are removed from a pile, clear the bounding box
' of the pile to erase the image of the removed cards.
' (Must be called before AdjustBoundingBox)
sub ClearBoundingBox which
  local x, y, w, h
  x = bboxes(1, which) : y = bboxes(2, which)
  w = bboxes(3, which)-x : h = bboxes(4, which)-y
  box x, y, w, h,, TABLE_COLOR, TABLE_COLOR
end sub

' Adjust the bounding box for a pile as cards are added/removed
sub AdjustBoundingBox pindex
  local h, x, y, xjog, yjog, jw, jh, nc
  local col, row
  col = pile_info(P_COL, pindex)-1
  row = pile_info(P_ROW, pindex)-1
  xjog = pile_info(P_ST_X, pindex)*CARD_HJOG
  yjog = pile_info(P_ST_Y, pindex)*CARD_VJOG
  if xjog >= 0 then
    x = LEFT_MARGIN + col*(CARD_WIDTH+CARD_SPACING)
  else
    x = LEFT_MARGIN + col*(CARD_WIDTH+CARD_SPACING) + CARD_WIDTH
  end if
  y = TOP_MARGIN + row*(CARD_HEIGHT+CARD_VSPACING)
  nc = pnum_cards(pindex)
  if nc > 1 then
    jw = (nc-1)*xjog
    jh = (nc-1)*yjog
  else
    jw = 0 : jh = 0
  end if
  w = CARD_WIDTH + abs(jw)
  h = CARD_HEIGHT + jh
  if xjog >= 0 then
    bboxes(1, pindex) = x
    bboxes(3, pindex) = x+w
  else
    bboxes(1, pindex) = x-w
    bboxes(3, pindex) = x
  end if
  bboxes(2, pindex) = y
  bboxes(4, pindex) = y+h  
end sub

' Flip the top card of a pile to the specified face (if there are any cards)
' If face < 0 then the card is flipped; otherwise it is turned to the requested face.
sub TurnTopCard pindex, face
  local nc, card, rank, suit, eface
  nc = pnum_cards(pindex)
  if nc > 0 then
    card = piles(nc, pindex)
    DecodeCard card, rank, suit, eface
    if face < 0 then
      face = 1 - eface
    end if
    card = EncodeCard(rank, suit, face)
    piles(nc, pindex) = card
    DrawPile pindex
  end if
end sub

' Deal cards and recycle the discard pile back into the deck
' when needed.
sub DealCardsFromDeck
  local i, nd, nk, nc, card, dx, dp, flag
  nd = pnum_cards(deck_pindex)
  nc = min(nd, pile_info(P_RM_N, deck_pindex))
  flag = 0
  if nd = 0 then
    if pile_info(P_DL_D, deck_pindex) > 0 then
      nk = pnum_cards(discard_pindex(1))
      dp = discard_pindex(1)
      for i = 0 to nk-1
        RemoveCardFromPile dp, -1, card
        AddCardToPile card, deck_pindex
        TurnTopCard deck_pindex, FACEDOWN
      next i
      flag = 1
    end if
    selected_pile = 0
  else
    dx = 1
    dp = discard_pindex(dx)
    if nc > 0 then flag = 1
    for i = 1 to nc
      RemoveCardFromPile deck_pindex, -1, card
      AddCardToPile card, dp
      TurnTopCard dp, FACEUP
      inc dx
      if dx > num_discards then
        dx = 1
      end if
      dp = discard_pindex(dx)
    next i
  end if
  if flag = 1 then
    DrawPile deck_pindex
    for i = 1 to num_discards
      DrawPile discard_pindex(i)
    next i
  end if
end sub

' returns a card encoded into an integer
function EncodeCard(rank, suit, face)
  EncodeCard = rank*100 + suit*10 + face
end function

' decodes an integer card into its components
sub DecodeCard card, rank, suit, face
  rank = card\100
  suit = (card-100*rank)\10
  face = card-100*rank-10*suit
end sub

' Enforce card removal rules (1 = ok, 0 = no)
' pindex = source pile index
' dindex = destination pile index (or zero)
' cindex = card index in pile
' ncards = number of cards to be removed
function CanRemove(sindex, dindex, cindex, ncards)
  local i, rank, suit, face, drole
  local nc = pnum_cards(sindex)
  local rn = pile_info(P_RM_N, sindex)
  if dindex > 0 then
    drole = pile_info(P_ROLE, dindex)
  else
    drole = 0
  end if
  if (nc < ncards) or ((rn > 0) and (rn < ncards)) then
    CanRemove = 0
    rm_reason = RR_NUM
    exit function
  end if
  if rn = RM_ALL_CARDS and ncards <> nc and drole = ROLE_TABLEAU then
    CanRemove = 0
    rm_reason = RR_NUM
    exit function
  end if
  local rf = pile_info(P_RM_F, sindex)
  for i = nc to nc-ncards+1 step -1
    DecodeCard piles(i, sindex), rank, suit, face
    if face <> rf then
      CanRemove = 0
      rm_reason = RR_FACE
      exit function
    end if
  next i
  rm_reason = 0
  CanRemove = 1
end function

' Enforce card add rules (1 = ok, 0 = no)
' pindex = destination pile index
' sindex = source pile index
' cindex = start card index in source pile
' ncards = number of cards to be added
function CanAdd(pindex, sindex, cindex, ncards)
  local card, rank, suit, face, irank, adf, adr, ads, adw
  local ok, prank, psuit, pface, mxcards, mxadd, fcard
  local nsc = pnum_cards(sindex)
  if nsc < ncards then 
    CanAdd = 0
    add_reason = AR_NOCARDS
    exit function
  end if
  fcard = piles(nsc-ncards+1, sindex)
  local ndc = pnum_cards(pindex)
  mxcards = pile_info(P_AD_M, pindex)
  if (mxcards >= 0) and (ndc+ncards > mxcards) then
    CanAdd = 0
    add_reason = AR_MAXTOTAL
    exit function
  end if
  mxcards = pile_info(P_AD_N, pindex)
  if (mxcards >= 0) and (ncards > mxcards) then
    CanAdd = 0
    add_reason = AR_BATCHSIZE
    exit function
  end if
  if ndc = 0 then
    DecodeCard fcard, rank, suit, face
    irank = pile_info(P_AD_I, pindex)
    if irank> 0 and rank <> irank then
      CanAdd = 0
      add_reason = AR_INITRANK
      exit function
    end if
  end if
  ok = 1
  if pnum_cards(pindex) > 0 then
    card = piles(ndc, pindex)
    DecodeCard card, prank, psuit, pface
    DecodeCard fcard, rank, suit, face
    adf = pile_info(P_AD_F, pindex)
    adr = pile_info(P_AD_R, pindex)
    ads = pile_info(P_AD_S, pindex)
    adw = pile_info(P_AD_W, pindex)
    if adf <> -1 then
      if face <> adf then
        ok = 0
        add_reason = AR_FACE
      end if
    end if
    if adr < 0 then
      if rank <> WrapRank(prank, -1, adw) then
        ok = 0
        add_reason = AR_DECRANK
      end if
    else if adr = 99 then
      if rank <> WrapRank(prank, 1, adw) then
        if rank <> WrapRank(prank, -1, adw) then
          ok = 0
          add_reason = AR_DECRANK
        end if
      end if
    else if adr > 0 then
      if rank <> WrapRank(prank, 1, adw) then
        ok = 0
        add_reason = AR_INCRANK
      end if
    else if (solitaire_type = PILE_ON) and (adr = 0) then
      if rank <> prank then
        ok = 0
        add_reason = AR_SAMERANK
      end if
    end if
  end if
  if ads <> 0 then
    if ads = 1 then
      if suit <> psuit then
        ok = 0
        add_reason = AR_SUIT
      end if
    else if ads = 2 then
      if (psuit = HEARTS) or (psuit = DIAMONDS) then
        if (suit <> SPADES) and (suit <> CLUBS) then
          ok = 0
          add_reason = AR_ALTCRB
        end if
      else
        if (suit <> HEARTS) and (suit <> DIAMONDS) then
          ok = 0
          add_reason = AR_ALTCBR
        end if
      end if
    end if
  end if
  if ok then add_reason = 0
  CanAdd = ok
end function

' Handle Wrapped or unwrapped rank comparison
function WrapRank(rank, diff, wflag)
  if not wflag then
    WrapRank = rank+diff
  else
    if rank = ACE and diff < 0 then
      WrapRank = KING
    else if rank = KING and diff > 0 then
      WRapRank = ACE
    else
      WrapRank = rank+diff
    end if
  end if
end function

' Select the source or destination pile on mouse click
' If possible, move the selected cards from source pile 
' to destination pile.
' note: only handing card index for verital jogs!
sub SelectPile which, my
  local nstate = 0
  local nsc, ndc, yd, cx, yJog, i
  if destination_pile > 0 then
    ClearMovePiles
    exit sub
  end if
  select case move_state
    case 0
      source_pile = which
      HilitePile which, SELECTED_COLOR
      nstate = 1
    case 1
      destination_pile = which
      nc = GetNumCardsToMove(source_pile, prev_left_mx, prev_left_my)
      if CanRemove(source_pile, destination_pile, nsc, nc) then
        nsc = pnum_cards(source_pile)
        cx = nsc-nc+1
        if CanAdd(destination_pile, source_pile, cx, nc) then
          MoveCards source_pile, destination_pile, nc
        else
          nstate = 0
        end if
      else
        nstate = 0
      end if
      DrawPile source_pile
      DrawPile destination_pile
      ClearMovePiles
  end select
  move_state = nstate          
end sub

' After validation has been done, move the requested cards from
' the source to destination piles.
sub MoveCards source_pile, destination_pile, nc
  local rule, nsc, cx, i, card
  role = pile_info(P_ROLE, source_pile)
  nsc = pnum_cards(source_pile)
  cx = nsc-nc+1
  for i = nc to 1 step -1
    RemoveCardFromPile source_pile, cx, card
    AddCardToPile card, destination_pile
  next i
  if CheckWin() then ShowWin
  DoSpecialPostmoveRules
end sub

' Based on the location of the mouse click relative to the top of the
' source pile, return the number of cards selected to move
function GetNumCardsToMove(which, mx, my)
  local xJog, yJog, sx, nsc, yd, cx, nc, axj
  xJog = pile_info(P_ST_X, which)*CARD_HJOG
  yJog = pile_info(P_ST_Y, which)*CARD_VJOG
  nsc = pnum_cards(which)
  if yJog > 0 and nsc > 1 then
    yd = my - bboxes(2, which)
    cx = min(nsc, (yd+yJog-1)\yJog)
    nc = nsc-cx+1
  else if xJog > 0 and nsc > 1 then
    xd = mx - bboxes(1, which)
    cx = min(nsc, (xd+xJog-1)\xJog)
    nc = nsc-cx+1
  else if xJog < 0 and nsc > 1 then
    sx = bboxes(1, which) + CARD_WIDTH
    if mx > sx then
      xd = abs(mx - sx)
      axj = abs(xJog)
      cx = nsc - min(nsc, (xd+axj-1)\axj)
      nc = nsc-cx+1
    else
      nc = 1
    end if
  else
    nc = 1
  end if
  GetNumCardsToMove = nc
end function

' Hilite a pile with the specified edge color
sub HilitePile pindex, c
  local x, y, w, h
  x = bboxes(1, pindex) : y = bboxes(2, pindex)
  w = bboxes(3, pindex) - x
  h = bboxes(4, pindex) - y
  rbox x, y, w, h + (nc-1)*cstep, 10, c
end sub

' clear the designators and hilites for both the
' source and destination piles
sub ClearMovePiles
  if source_pile > 0 then
    HilitePile source_pile, UNSELECTED_COLOR
  end if
  source_pile = -1
  if destination_pile > 0 then
    HilitePile destination_pile, UNSELECTED_COLOR
  end if
  destination_pile = -1
  move_state = 0
end sub

' returns 1 if game is won, 0 otherwise
function CheckWin()
  local i, j, win, sf, ef, card, rank, suit, face, rank2
  win = 1
  if solitaire_type = PILE_ON then
    CheckWin = CheckPileOnWin()
    exit function
  else if solitaire_type = GOLF then
    CheckWin = CheckGolfWin()
    exit function
  end if
  for i = 1 to num_piles
    if pile_info(P_ROLE, i) = ROLE_FOUNDATION then
      sf = i
      exit for
    end if
  next i
  ef = sf + num_foundations - 1
  for i = sf to ef
    if pnum_cards(i) <> NRANKS then win = 0
  next i
  CheckWin = win
end function

' Special check for a won game with PILEON game
function CheckPileOnWin()
  local win, i, j, card, rank, rank2, suit, face
  win = 1
  for i = 2 to num_piles
    if pnum_cards(i) <> 4 and pnum_cards(i) <> 0 then
      CheckPileOnWin = 0
      exit function
    end if
    if pnum_cards(i) > 0 then
      card = piles(1, i)
      DecodeCard card, rank, suit, face
      for j = 2 to pnum_cards(i)
        card = piles(j, i)
        DecodeCard card, rank2, suit, face
        if rank2 <> rank then
          CheckPileOnWin = 0
          exit function
        end if
      next j
    end if
  next i
  CheckPileOnWin = win
end function

' Special check for a won game with GOLF game
function CheckGolfWin()
  local i, tx, win
  for i = 1 to num_piles
    if pile_info(P_ROLE, i) = ROLE_TABLEAU then
      tx = i
      exit for
    end if
  next i
  win = 1
  for i = tx to tx+num_tableaux-1
    if pnum_cards(i) > 0 then
      win = 0
      exit for
    end if
  next i
  CheckGolfWin = win
end function
    
' Show a won game
sub ShowWin
  local x, y, w, h
  running = 0
  gui cursor hide
  x = 100: y = 200 : w = 600 : h = 100
  box x, y, w, h, 2, rgb(red), rgb(black)
  text x+w\2, y+h\2, "YOU WIN!!!", "CM", 5, 2, rgb(blue), -1
  text x+w\2, y+h-10, "Press Any Key to Continue", "CB",,, rgb(blue), -1
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  DrawTable
end sub

' Do special rules specific to each solitaire type
sub DoSpecialPostmoveRules
  select case solitaire_type
    case KLONDIKE
    case CANFIELD
      DoCanfieldPostmoveSpecial
    case EIGHTOFF
    case SCORPION
    case YUKON
  end select
end sub

' Do the special postmove rule for Canfield:
' Fill empty tableau pile from reserve
sub DoCanfieldPostmoveSpecial
  local rindex, i, rnc, card
  rindex = 11
  rnc = pnum_cards(rindex)
  for i = 1 to num_piles
    if pile_info(P_ROLE, i) = ROLE_TABLEAU then
      if pnum_cards(i) = 0 and rnc > 0 then
        RemoveCardFromPile rindex, -1, card
        AddCardToPile(card, i)
        DrawPile i
        DrawPile rindex
        exit for
      end if
    end if
  next i
end sub

  ' Fetch the card image indices from the PNG file.
  ' Images are stored 10 across and 6 vertically
  ' Each card cell is 80 wide by 100 tall, the actual
  ' card graphics are 74 wide by 94 tall
sub GetCardImages
  page write 1
  cls
  load png "playing_cards_deck.png",0,0
  for i = 0 to 9
    source(i+1,1) = 3 + 80 * i    ' spades 1 to 10
    source(i+1,2) = 3             ' same y coordinate for all
    source(i+14,1) = 3 + 80 * i   ' hearts 1 to 10
    source(i+14,2) = 103
    source(i+27,1) = 3 + 80 * i   ' diamonds 1 to 10
    source(i+27,2) = 203
    source(i+40,1) = 3 + 80 * i   ' clubs 1 to 10
    source(i+40,2) = 303
  next i
  for i = 0 to 2
    source(i+11,1) = 3 + 80 * i   ' spades J Q K
    source(i+11,2) = 403
    source(i+24,1) = 3 + 80 * i   ' hearts J Q K
    source(i+24,2) = 503
    source(i+37,1) = 243 + 80 * i ' diamonds J Q K
    source(i+37,2) = 403
    source(i+50,1) = 243 + 80 * i ' clubs J Q K
    source(i+50,2) = 503
  next i
  for i = 0 to 1
    source(i+53,1) = 643 + 80 * i ' card backs
    source(i+53,2) = 403
  next i
  for i = 0 to 1
    source(i+55,1) = 643 + 80 * i ' card backs
    source(i+55,2) = 503
  next i 
  page write 0
end sub

' Draw the Card table and all the card piles
sub DrawTable
  local ploc, row, col, jog, x, y, i
  local nc, card
  box 0, 0, 799, 599,, TABLE_COLOR, TABLE_COLOR  
  for i = 1 to num_piles
    DrawPile i    
  next i
  DrawButtons
end sub

' Draw a Pile
sub DrawPile which
  local i, row, col, px, py, pw, ph, xjog, yjog, nc, x, y
  local w, h, irank
  gui cursor hide
  px = bboxes(1, which) : py = bboxes(2, which)
  pw = bboxes(3, which)-px : ph = bboxes(4, which) - py
  if pile_info(P_COL, which) = 0 then
    gui cursor show
    exit sub
  end if
  xjog = pile_info(P_ST_X, which)*CARD_HJOG
  yjog = pile_info(P_ST_Y, which)*CARD_VJOG
  rbox px, py, pw, ph, CARD_CRAD
  nc = pnum_cards(which)
  if xjog >= 0 then
    x = px
  else
    x = px - (nc-1)*xjog
  end if
  y = py
  for i = 1 to nc
    DrawCard piles(i, which), x, y
    inc x, xjog
    inc y, yjog
  next i
  w = bboxes(3, which)-px
  h = bboxes(4, which)-py
  irank = pile_info(P_AD_I, which)
  if nc = 0 then
    if which = deck_pindex and pile_info(P_DL_D, deck_pindex) > 0 then
      text x+w\2, y+h\2-10, "Click to", "CM", 7,, rgb(white), -1
      text x+w\2, y+h\2+10, "Turm Over", "CM", 7,, rgb(white), -1
    end if
    if irank > 0 then
      select case irank
        case 1  : r$ = "A"
        case 2 to 10 : r$ = str$(irank)
        case 11 : r$ = "J"
        case 12 : r$ = "Q"
        case 13 : r$ = "K"
      end select
      text px+5, py+5, r$, "LT", 7,, rgb(white), -1
    end if
  end if
  role = pile_info(P_ROLE, which)
  if role = ROLE_DECK or role = ROLE_DISCARD then
    nc = pnum_cards(which)
    c$ = str$(nc)
    box px, py+h+3, 15, 15,, TABLE_COLOR, TABLE_COLOR
    text px, py+h+3, c$, "LT", 7,, rgb(white), -1
  end if
  if role = ROLE_RESERVE and (solitaire_type = EIGHTOFF or solitaire_type = FREECELL) then
    if nc = 0 then
      text x+w\2, y+h\2-10, "Free", "CM", 7,, rgb(white), -1
    end if
  end if
  gui cursor show
end sub

' Render a specified card
sub DrawCard card, x, y
  local index, rank, suit, face
  DecodeCard card, rank, suit, face
  if face = FACEUP then
    index = (suit-1)*NRANKS + rank
  else
    index = CBACK_SOURCE_INDEX
  end if
  blit source(index, 1), source(index, 2), x, y, CARD_WIDTH, CARD_HEIGHT, 1, 4
end sub

' Draw the control buttons
sub DrawButtons
  local x, y, w, h, i
  local m$
  w = button_w
  h = button_h
  y = button_y
  for i = 1 to NBUTTONS
    x = button_x(i)
    rbox x, y, w, h, 5, rgb(black)
    select case i
      case 1 : m$ = "Restart"
      case 2 : m$ = "New Game"
      case 3 : m$ = "Games Menu"
      case 4 : m$ = "Rules"
      case 5 : m$ = "Quit"
    end select
    text x+w\2, y+h\2, m$, "CM",,, rgb(black), -1
  next i
end sub

' Detect mouse left clicks in virtual command buttons at screen top
function HandleButtons(mx, my)
  local i, hit
  hit = 0
  for i = 1 to NBUTTONS
    if my >= button_y and my < button_y+button_h then
      if mx >= button_x(i) and mx < button_x(i)+button_w then
        DoCommand i
        hit = 1
        exit for
      end if
    end if
  next i
  HandleButtons = hit
end function

' Handle commands from virtual buttons
sub DoCommand which
  select case which
    case 1: RestartGame
    case 2: StartNewGame
    case 3: ShowGameMenu
    case 4: ShowRules
    case 5: Quit
  end select
end sub

' Restart the current game
sub RestartGame
  local i
  for i = 1 to num_piles
    pnum_cards(i) = 0
  next i
  for i = 1 to NCARDS
    piles(i, deck_pindex) = restart_deck(i)
  next i
  pnum_cards(deck_pindex) = NCARDS
  DealInitialCards
  ClearMovePiles
  running = 1
  DrawTable
end sub

' start a new game of the current type
sub StartNewGame
  ClearMovePiles
  NewGame
  DrawTable
  running = 1
  sp = 0
end sub

' Show the Menu of available games and let the user pick using arrow keys
' and ENTER.
sub ShowGameMenu
  local x, y, w, h, nc, nr, gx
  cls
  gui cursor hide
  gs_active = 1
  running = 0
  text mm.hres\2, 20, "Available Solitaire Types", "CT", 5,, rgb(green)
  text mm.hres\2, 60, "Use arrow keys to navigate, ENTER to pick a game type", "CT"
  text mm.hres\2, 80, "Press the F1 key for general instructions", "CT"
  w = GS_WIDTH : h = GS_HEIGHT
  x = GS_X
  y = GS_Y
  nc = 2
  nr = MAX_SOLS\2
  gx = 1
  for gs_row = 1 to nr
    box x, y, w, h,, rgb(blue), 2
    text x+w\2, y+h\2, snames$(gx), "CM", 4,, rgb(green)
    inc y, h
    inc gx
  next gs_row
  x = mm.hres\2
  y = GS_Y
  for gs_row = 1 to nr
    box x, y, w, h,, rgb(blue), 2
    text x+w\2, y+h\2, snames$(gx), "CM", 4,, rgb(green)
    inc y, h
    inc gx
  next gs_row
  gs_col = 1 : gs_row = 1
  HiliteGame gs_row, gs_col
end sub

' Hilite a game type box in yellow
sub HiliteGame row, col
  local x, y
  static prev_row = 0
  static prev_col = 0
  if prev_row > 0 then
    if prev_col = 1 then 
      x = GS_X
    else
      x = mm.hres\2
    end if
    y = GS_Y + (prev_row-1)*GS_HEIGHT
    box x, y, GS_WIDTH, GS_HEIGHT, 2, rgb(blue)
  end if
  if col = 1 then 
    x = GS_X
  else
    x = mm.hres\2
  end if
  y = GS_Y + (row-1)*GS_HEIGHT
  box x, y, GS_WIDTH, GS_HEIGHT, 2, rgb(yellow)
  prev_row = row : prev_col = col
end sub

' Show the start screen with games menu
sub ShowInstructions
  local x = 60
  cls
  text mm.hres\2, 10, "Solitaire", "CT", 5,, rgb(green)
  text 0, 60, ""
  print @(x);"This program allows you to play several different kinds of solitaire."
  print ""
  print @(x);"To move cards between one pile and another, left-click the pile you want"
  print @(x);"to move cards FROM, and then left-click the pile you want to move cards"
  print @(x);"TO. If the move is legal, the cards will be moved.
  print ""
  print @(x);"It makes a difference where you click on a FROM pile. If you click on the"
  print @(x);"card that is fully visible (the top card of the pile), then you will move"
  print @(x);"just that one card. If you click on one of the other face-up cards that are"
  print @(x);"farther down in the pile (higher on the screen), then more cards will be moved,"
  print @(x);"if the move is legal. And clicking on a face-down card doesn't do anything."
  print ""
  print @(x);"After choosing the FROM pile with a left-click, you can speed up the move"
  print @(x);"by right-clicking the same pile. If there is a legal move, it will be done."
  print @(x);"Moves to foundations have priority over other moves. Be careful, the default"
  print @(x);"move may not always be what you want if there are multiple possibilities!"
  print @(x);""
  print @(x);"You can restart a game using the Restart button at the top of the screen."
  print @(x);"Other buttons let you start a new game of the same type, return to the menu"
  print @(x);"of game types, or quit the program.
  print @(x);""
  print @(x);"To read the rules for a game, click the Rules button at the screen top."
  text mm.hres\2, 500, "Press Any Key to Return", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
end sub

' Show rules for the current game type
sub ShowRules
  local i, y
  local m$
  box 10, 225, 780, 150, 2, rgb(green), rgb(black)
  y = 245
  m$ = "Rules for " + snames$(solitaire_type)
  text mm.hres\2, 230, m$, "CT",,, rgb(yellow)
  text 20, 250, ""
  for i = 1 to num_rule_lines
    print @(20, y);rules$(i)
    inc y, 15
  next i
  text mm.hres\2, 370, "Press Any Key to Continue", "CB"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  DrawTable
end sub

' (debug)
sub PrintSolitaireDef
  local p
  print #1, "Solitaire Type: ";solitaire_type
  print #1, "Number of decks: ";num_decks
  print #1, "Number of Foundations: ";num_foundations
  print #1, "Number of Tableaux: ";num_tableaux
  print #1, "Number of Reserves: ";num_reserves
  print #1, "Total Piles:        ";num_piles
  print #1, ""
  print #1, "Pile Role  Layout      Remove   Add                     Predeal"
  for p = 1 to num_piles
    print #1, format$(p, "%2g");"    ";pile_info(1, p);
    print #1, "  ";pile_info(2, p);",";pile_info(3,p);",";pile_info(4,p);",";pile_info(5,p);
    print #1, "  ";pile_info(6,p);",";pile_info(7,p);
    print #1, "    ";pile_info(8,p);","pile_info(9,p);",";pile_info(10,p);",";pile_info(11,p);",";
    print #1, pile_info(12,p);","pile_info(13,p);",";pile_info(14,p);
    print #1, "  ";pile_info(15, p);",";pile_info(16,p)
  next p
end sub

data "Klondike", "Canfield", "EightOff", "PileOn",      "Yukon"
data "Golf", "Bristol",  "FreeCell", "FourSeasons", "BeleaguredCastle"

'----- End Source Code









